data "aws_region" "current" {
}

resource "kubernetes_deployment" "aws_cluster_autoscaler_deployment" {
  metadata {
    name      = "aws-cluster-autoscaler"
    namespace = "kube-system"

    labels = {
      app = "aws-cluster-autoscaler"
    }
  }

  spec {
    replicas = 1

    selector {
      match_labels = {
        app      = "aws-cluster-autoscaler"
        instance = "aws-cluster-autoscaler-deploy"
      }
    }

    template {
      metadata {
        labels = {
          app      = "aws-cluster-autoscaler"
          instance = "aws-cluster-autoscaler-deploy"
        }

        annotations = {
          "iam.amazonaws.com/role" = "eks-cluster-autoscaler"
        }
      }

      spec {
        service_account_name             = "cluster-autoscaler-svc-acc"
        termination_grace_period_seconds = 60

        volume {
          name = kubernetes_service_account.cluster_autoscaler_service_account.default_secret_name

          secret {
            secret_name = kubernetes_service_account.cluster_autoscaler_service_account.default_secret_name
          }
        }

        volume {
          name = "autoscaler-ssl-volume"
          empty_dir {
          }
        }

        container {
          image             = "${var.cluster_autoscaler_image_region}.gcr.io/k8s-artifacts-prod/autoscaling/cluster-autoscaler:v${var.cluster_autoscaler_version}"
          name              = "cluster-autoscaler-container"
          image_pull_policy = "IfNotPresent"
          args              = ["./cluster-autoscaler", "--cloud-provider=aws", "--node-group-auto-discovery=asg:tag=k8s.io/cluster-autoscaler/enabled", "--skip-nodes-with-local-storage=false", "--stderrthreshold=info", "--expander=least-waste", "--v=4"]

          volume_mount {
            mount_path = "/var/run/secrets/kubernetes.io/serviceaccount"
            name       = kubernetes_service_account.cluster_autoscaler_service_account.default_secret_name
            read_only  = true
          }

          volume_mount {
            mount_path = "/etc/ssl/certs/ca-bundle.crt"
            name       = "autoscaler-ssl-volume"
            read_only  = true
          }

          resources {
            limits {
              cpu    = "100m"
              memory = "300Mi"
            }

            requests {
              cpu    = "100m"
              memory = "300Mi"
            }
          }

          port {
            container_port = 8085
          }

          env {
            name  = "AWS_REGION"
            value = data.aws_region.current.name
          }
        }
      }
    }
  }
}

resource "kubernetes_service" "cluster_autoscale_service" {
  metadata {
    name = "cluster-autoscaler"
  }

  spec {
    port {
      port        = 8085
      protocol    = "TCP"
      target_port = 8085
    }

    selector = {
      app = "aws-cluster-autoscaler"
    }

    type = "ClusterIP"
  }
}

