#############################################################################################################
# Variables
#############################################################################################################
variable "vpc_id" {}
variable "security_groups" {
  type = "list"
}

variable "subnet_ids" {
  type = "list"
}

variable "rds_instance_name" {}

variable "rds_engine" {
  default = "mysql"
}

variable "rds_engine_version" {
  default = "5.6.27"
}

variable "rds_tcp_port" {
    default = "3306"
}

variable "rds_allocated_storage" {
  description = "The allocated storage in GBs"
  default = "100"
}

variable "rds_storage_type" {
  default = "gp2"
}

variable "rds_instance_class" {
  default = "db.t2.micro"
}

variable "rds_username" {}
variable "rds_password" {}

variable "rds_multi_az" {
  default = "true"
}

variable "rds_publicly_accessible" {
  default = "false"
}

variable "backup_retention_period" {
  default = "7"
}

variable "backup_window" {
  default = "00:00-01:00"
}

variable "maintenance_window" {
  default = "Sun:01:00-Sun:02:00"
}

variable "read_replica_count" {
  default = "0"
}

variable "parameter_group" {
    default = "default.mysql5.6"
}

variable "route53_zone" {}

variable "quicksight_rds_ingress_from_port" {
  default = "3306"
}

variable "quicksight_rds_ingress_to_port" {
  default = "3306"
}

variable "quicksight_rds_egress_from_port" {
  default = "3306"
}

variable "quicksight_rds_egress_to_port" {
  default = "3306"
}

variable "quicksight_ingress_from_port" {
  default = "0"
}

variable "quicksight_ingress_to_port" {
  default = "65535"
}

variable "quicksight_egress_from_port" {
  default = "0"
}

variable "quicksight_egress_to_port" {
  default = "65535"
}

#############################################################################################################
# Security Group
#############################################################################################################

resource "aws_security_group" "rds" {
  vpc_id      = "${var.vpc_id}"
  description = "${var.rds_instance_name} RDS SG"

  ingress {
    from_port       = "${var.rds_tcp_port}"
    to_port         = "${var.rds_tcp_port}"
    protocol        = "tcp"
    security_groups = ["${var.security_groups}"]
  }
}

#############################################################################################################
# Subnet Group
#############################################################################################################

resource "aws_db_subnet_group" "default" {
  name             = "${var.rds_instance_name}-subnet"
  description      = "RDS Subnet group for ${var.rds_instance_name}"
  subnet_ids       = ["${var.subnet_ids}"]
}

#############################################################################################################
# RDS Instances
#############################################################################################################
resource "aws_db_instance" "default" {
  identifier              = "${var.rds_instance_name}"
  engine                  = "${var.rds_engine}"
  engine_version          = "${var.rds_engine_version}"

  allocated_storage       = "${var.rds_allocated_storage}"
  storage_type            = "${var.rds_storage_type}"

  instance_class          = "${var.rds_instance_class}"

  username                = "${var.rds_username}"
  password                = "${var.rds_password}"

  multi_az                = "${var.rds_multi_az}"
  publicly_accessible     = "${var.rds_publicly_accessible}"
  vpc_security_group_ids  = ["${aws_security_group.rds.id}"]
  db_subnet_group_name    = "${aws_db_subnet_group.default.name}"

  parameter_group_name    = "${var.parameter_group}"

  backup_retention_period = "${var.backup_retention_period}"
  backup_window           = "${var.backup_window}"
  maintenance_window      = "${var.maintenance_window}"

}

resource "aws_db_instance" "read_replica" {
  identifier              = "${var.rds_instance_name}-read-replica"
  engine                  = "${var.rds_engine}"
  engine_version          = "${var.rds_engine_version}"

  allocated_storage       = "${var.rds_allocated_storage}"
  storage_type            = "${var.rds_storage_type}"

  instance_class          = "${var.rds_instance_class}"

  username                = "${var.rds_username}"
  password                = "${var.rds_password}"

  multi_az                = "${var.rds_multi_az}"
  publicly_accessible     = "${var.rds_publicly_accessible}"
  vpc_security_group_ids  = ["${aws_security_group.rds.id}"]
  db_subnet_group_name    = "${aws_db_subnet_group.default.name}"

  parameter_group_name    = "${var.parameter_group}"

  backup_retention_period = "${var.backup_retention_period}"
  backup_window           = "${var.backup_window}"
  maintenance_window      = "${var.maintenance_window}"

  replicate_source_db     = "${aws_db_instance.default.name}"
  count                   = "${var.read_replica_count}"
}

#############################################################################################################
# Route 53 Records
#############################################################################################################

resource "aws_route53_record" "private_rds_default" {
  zone_id     = "${var.route53_zone}"
  name        = "${var.rds_engine}.${var.rds_instance_name}.aws"
  type        = "CNAME"
  ttl         = "60"
  records     = ["${aws_db_instance.default.address}"]
  count       = "${length(split(",", "var.route53_zone"))}"
}

resource "aws_route53_record" "private_rds_read-replica" {
  zone_id     = "${var.route53_zone}"
  name        = "readonly-${var.rds_engine}.${var.rds_instance_name}.aws"
  type        = "CNAME"
  ttl         = "60"
  records     = ["${aws_db_instance.read_replica.address}"]
  count       = "${length(split(",", "var.route53_zone")) * var.read_replica_count}"
}

#############################################################################################################
# QuickSight to RDS Security Group
#############################################################################################################

resource "aws_security_group" "quicksight_rds" {
  name        = "QuickSight-RDS"
  description = "Allow QuickSight to RDS"
  vpc_id      = "${var.vpc_id}"
}

resource "aws_security_group_rule" "quicksight_rds_ingress" {
  type                     = "ingress"
  from_port                = "${var.quicksight_rds_ingress_from_port}"
  to_port                  = "${var.quicksight_rds_ingress_to_port}"
  protocol                 = "tcp"
  source_security_group_id = "${aws_security_group.quicksight.id}"
  security_group_id        = "${aws_security_group.quicksight_rds.id}"
}

resource "aws_security_group_rule" "quicksight_rds_egress" {
  type                     = "egress"
  from_port                = "${var.quicksight_rds_egress_from_port}"
  to_port                  = "${var.quicksight_rds_egress_to_port}"
  protocol                 = "tcp"
  source_security_group_id = "${aws_security_group.quicksight.id}"
  security_group_id        = "${aws_security_group.quicksight_rds.id}"
}

#############################################################################################################
# QuickSight Security Group
#############################################################################################################

resource "aws_security_group" "quicksight" {
  name        = "QuickSight"
  description = "QuickSight SG"
  vpc_id      = "${var.vpc_id}"
}

resource "aws_security_group_rule" "quicksight_ingress" {
  type                     = "ingress"
  from_port                = "${var.quicksight_ingress_from_port}"
  to_port                  = "${var.quicksight_ingress_to_port}"
  protocol                 = "tcp"
  source_security_group_id = "${aws_security_group.quicksight_rds.id}"
  security_group_id        = "${aws_security_group.quicksight.id}"
}

resource "aws_security_group_rule" "quicksight_egress" {
  type                     = "egress"
  from_port                = "${var.quicksight_egress_from_port}"
  to_port                  = "${var.quicksight_egress_to_port}"
  protocol                 = "tcp"
  source_security_group_id = "${aws_security_group.quicksight_rds.id}"
  security_group_id        = "${aws_security_group.quicksight.id}"
}

#############################################################################################################
# Outputs
#############################################################################################################
output "rds_instance_id" {
    value = "${aws_db_instance.default.id}"
}

output "rds_instance_address" {
    value = "${aws_db_instance.default.address}"
}

output "read_replica_rds_instance_id" {
    value = "${var.read_replica_count == "0" ? "null" : element(concat(aws_db_instance.read_replica.*.id, list("")), 0) }"
}

output "read_replica_rds_instance_address" {
    value = "${var.read_replica_count == "0" ? "null" : element(concat(aws_db_instance.read_replica.*.address, list("")), 0) }"
}

output "subnet_group_id" {
    value = "${aws_db_subnet_group.default.id}"
}

output "quicksight_rds" {
  value = "${aws_security_group.quicksight_rds.id}"
}

output "quicksight_rds_ingress" {
  value = "${aws_security_group_rule.quicksight_rds_ingress.id}"
}

output "quicksight_rds_ingress" {
  value = "${aws_security_group_rule.quicksight_rds_egress.id}"
}

output "quicksight" {
  value = "${aws_security_group.quicksight.id}"
}

output "quicksight_ingress" {
  value = "${aws_security_group_rule.quicksight_ingress.id}"
}

output "quicksight_ingress" {
  value = "${aws_security_group_rule.quicksight_egress.id}"
}
