#############################################################################################################
# Variables
#############################################################################################################
variable "name" {
  description = "Name of the ELB"
}

variable "vpc_id" {
  description = "ID of the VPC in which the ELB should reside"
}

variable "internal" {
  description = "Determines if the ELB is internal or not"
  default = false
}

variable "subnets" {
  type = "list"
}

variable "cross_zone_load_balancing" {
  default = "true"
}
variable "idle_timeout" {
  default = "60"
}
variable "connection_draining" {
  default = "true"
}
variable "connection_draining_timeout" {
  default = "30"
}

# Listener variables
variable "service_protocol" {
 description = "The protocol the backend service speaks"
 default = "http"
}

variable "service_port" {
  description = "The port the service on the EC2 instances listens on"
  default = "80"
}

variable "lb_port" {
  default = "443"
}
variable "lb_protocol" {
  default = "https"
}

variable "elb_cert" {}


# Health check variables
variable "health_check_healthy_threshold" {
  default = "2"
}
variable "health_check_unhealthy_threshold" {
  default = "2"
}
variable "health_check_timeout" {
  default = "3"
}
variable "health_check_target" {}
variable "health_check_interval" {
  default = "30"
}

variable "logs_enabled" {
  default = "false"
}

variable "logs_interval" {
  default = "5"
}

#############################################################################################################
# Security Group
#############################################################################################################

resource "aws_security_group" "main" {
  name        = "${var.name}-elb"
  description = "Allows traffic to elb from other security groups"
  vpc_id      = "${var.vpc_id}"

  ingress {
    from_port   = 80
    to_port     = 80
    protocol    = "tcp"
    cidr_blocks = ["0.0.0.0/0"]
  }

  ingress {
    from_port   = 443
    to_port     = 443
    protocol    = "tcp"
    cidr_blocks = ["0.0.0.0/0"]
  }

  egress {
    from_port   = 0
    to_port     = 0
    protocol    = -1
    cidr_blocks = ["0.0.0.0/0"]
  }
}

#############################################################################################################
# ELB
#############################################################################################################

resource "aws_elb" "main" {
  name                        = "${var.name}"
  internal                    = "${var.internal}"

  subnets                     = ["${var.subnets}"]
  security_groups             = ["${aws_security_group.main.id}"]
  cross_zone_load_balancing   = "${var.cross_zone_load_balancing}"

  idle_timeout                = "${var.idle_timeout}"
  connection_draining         = "${var.connection_draining}"
  connection_draining_timeout = "${var.connection_draining_timeout}"

  listener {
    lb_port           = "80"
    lb_protocol       = "http"
    instance_port     = "${var.service_port}"
    instance_protocol = "${var.service_protocol}"
  }

  listener {
    lb_port           = "443"
    lb_protocol       = "https"
    ssl_certificate_id = "${var.elb_cert}"
    instance_port     = "${var.service_port}"
    instance_protocol = "${var.service_protocol}"
  }

  access_logs {
    enabled  = "${var.logs_enabled}"
    bucket   = "${var.name}-${random_string.bucket_name.result}-elb-logs"
    interval = "${var.logs_interval}"
  }

  health_check {
    healthy_threshold   = "${var.health_check_healthy_threshold}"
    unhealthy_threshold = "${var.health_check_unhealthy_threshold}"
    timeout             = "${var.health_check_timeout}"
    target              = "${var.health_check_target}"
    interval            = "${var.health_check_interval}"
  }
}


#############################################################################################################
# S3 Bucket for Log Exports
#############################################################################################################
resource "random_string" "bucket_name" {
  length  = 8
  special = false
  upper   = false
}

data "aws_elb_service_account" "main" {}

resource "aws_s3_bucket" "elb_logs" {
  count = "${var.logs_enabled == "true" ? 1 : 0}"
  bucket = "${var.name}-${random_string.bucket_name.result}-elb-logs"
  acl    = "private"

  policy = <<POLICY
{
  "Id": "Policy",
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": [
        "s3:PutObject"
      ],
      "Effect": "Allow",
      "Resource": "arn:aws:s3:::${var.name}-${random_string.bucket_name.result}-elb-logs/AWSLogs/*",
      "Principal": {
        "AWS": [
          "${data.aws_elb_service_account.main.arn}"
        ]
      }
    }
  ]
}
POLICY
}

#############################################################################################################
# Outputs
#############################################################################################################
output "id" {
  value = "${aws_elb.main.id}"
}
output "name" {
  value = "${aws_elb.main.name}"
}
output "dns_name" {
  value = "${aws_elb.main.dns_name}"
}
output "zone_id" {
  value = "${aws_elb.main.zone_id}"
}
output "security_group" {
  value = "${aws_security_group.main.id}"
}
