resource "aws_launch_configuration" "workers" {
  associate_public_ip_address = false
  ebs_optimized               = "${var.ebs_optimized_workers}"
  iam_instance_profile        = "${aws_iam_instance_profile.workers.name}"
  image_id                    = "${var.ami_id != "" ? var.ami_id : data.aws_ami.eks_worker.id}"
  instance_type               = "${var.instance_type}"
  name_prefix                 = "${var.cluster_name}-eks-workers-"
  security_groups             = ["${aws_security_group.workers.id}"]
  user_data_base64            = "${base64encode(data.template_file.workers_userdata.rendered)}"

  lifecycle {
    create_before_destroy = true
  }
}

resource "aws_autoscaling_group" "workers" {
  desired_capacity     = "${var.desired_workers}"
  launch_configuration = "${aws_launch_configuration.workers.id}"
  max_size             = "${var.max_workers}"
  min_size             = "${var.min_workers}"
  name                 = "${var.cluster_name}-eks-workers - ${aws_launch_configuration.workers.name}"
  vpc_zone_identifier  = ["${var.private_subnets}"]

  tag {
    key                 = "Name"
    value               = "${var.cluster_name}-eks-workers"
    propagate_at_launch = true
  }

  tag {
    key                 = "kubernetes.io/cluster/${var.cluster_name}"
    value               = "owned"
    propagate_at_launch = true
  }

  tag {
    key                 = "k8s.io/cluster-autoscaler/enabled"
    value               = "true"
    propagate_at_launch = true
  }

  tag {
    key                 = "k8s.io/cluster-autoscaler/${var.cluster_name}"
    value               = "owned"
    propagate_at_launch = true
  }

  lifecycle {
    ignore_changes = ["desired_capacity"]
  }
}

resource "aws_security_group" "workers" {
  name        = "${var.cluster_name}-eks-workers"
  description = "Security group for all EKS worker nodes in the cluster"
  vpc_id      = "${var.vpc_id}"

  egress {
    from_port   = 0
    to_port     = 0
    protocol    = "-1"
    cidr_blocks = ["0.0.0.0/0"]
  }

  tags = "${
    map(
     "Name", "${var.cluster_name}-eks-workers",
     "kubernetes.io/cluster/${var.cluster_name}", "owned",
    )
  }"
}

resource "aws_security_group_rule" "workers_ingress" {
  description              = "Allow EKS worker nodes to communicate with each other"
  from_port                = 0
  protocol                 = "-1"
  security_group_id        = "${aws_security_group.workers.id}"
  source_security_group_id = "${aws_security_group.workers.id}"
  to_port                  = 65535
  type                     = "ingress"
}

resource "aws_security_group_rule" "workers_allow_masters" {
  description              = "Allow EKS worker node Kubelets and pods to receive communication from the cluster control plane"
  from_port                = 1025
  protocol                 = "tcp"
  security_group_id        = "${aws_security_group.workers.id}"
  source_security_group_id = "${aws_security_group.masters.id}"
  to_port                  = 65535
  type                     = "ingress"
}

resource "aws_iam_role" "workers" {
  name = "${var.cluster_name}-eks-workers"

  assume_role_policy = "${data.aws_iam_policy_document.workers.json}"
}

resource "aws_iam_role_policy_attachment" "workers_node_policy" {
  policy_arn = "arn:aws:iam::aws:policy/AmazonEKSWorkerNodePolicy"
  role       = "${aws_iam_role.workers.name}"
}

resource "aws_iam_role_policy_attachment" "workers_cni_policy" {
  policy_arn = "arn:aws:iam::aws:policy/AmazonEKS_CNI_Policy"
  role       = "${aws_iam_role.workers.name}"
}

resource "aws_iam_role_policy_attachment" "workers_ecr_registry" {
  policy_arn = "arn:aws:iam::aws:policy/AmazonEC2ContainerRegistryReadOnly"
  role       = "${aws_iam_role.workers.name}"
}

resource "aws_iam_instance_profile" "workers" {
  name = "${var.cluster_name}-eks-workers-instance-profile"
  role = "${aws_iam_role.workers.name}"
}

resource "aws_iam_policy" "workers_autoscaling" {
  name   = "${var.cluster_name}-eks-kubernetes-workers-autoscaling"
  policy = "${data.aws_iam_policy_document.workers_autoscaling.json}"
}

resource "aws_iam_role_policy_attachment" "workers_autoscaling" {
  role       = "${aws_iam_role.workers.name}"
  policy_arn = "${aws_iam_policy.workers_autoscaling.arn}"
}
