data "aws_region" "current" {
  current = true
}

data "aws_ami" "ami" {
  most_recent = true

  filter {
    name   = "name"
    values = ["${var.ami_name_pattern}"]
  }

  filter {
    name   = "virtualization-type"
    values = ["hvm"]
  }

  filter {
    name   = "root-device-type"
    values = ["ebs"]
  }

  owners = ["${var.ami_owner}"]
}

data "template_file" "userdata" {
  template = "${file("${path.module}/userdata.tpl")}"

  vars {
    name                = "${var.name}"
    efs_mount_target    = "${var.efs_mount_target}"
    s3_bucket_path      = "${var.s3_bucket_path == "" ? join("", aws_s3_bucket.efs_backup.*.id) : var.s3_bucket_path}"
    backup_archive_name = "${var.backup_archive_name}"
    region              = "${data.aws_region.current.name}"
  }
}

resource "aws_launch_configuration" "backup_efs_node" {
  lifecycle {
    create_before_destroy = true
  }

  name_prefix          = "${var.name}"
  image_id             = "${data.aws_ami.ami.id}"
  instance_type        = "${var.instance_type}"
  security_groups      = ["${var.security_groups}"]
  iam_instance_profile = "${aws_iam_instance_profile.backup_efs_node.arn}"
  key_name             = "${var.key_name}"
  user_data            = "${data.template_file.userdata.rendered}"

  root_block_device {
    volume_size = "${var.root_volume_size}"
  }
}

resource "aws_autoscaling_group" "backup_efs_node" {
  name                 = "${var.name}"
  max_size             = 1
  min_size             = 0
  desired_capacity     = 0
  launch_configuration = "${aws_launch_configuration.backup_efs_node.name}"
  vpc_zone_identifier  = ["${var.subnet_ids}"]

  tags {
    key                 = "Name"
    value               = "${var.name}"
    propagate_at_launch = true
  }
}

resource "aws_autoscaling_schedule" "backup_efs_node" {
  scheduled_action_name  = "${var.name}-scale-up"
  min_size               = 0
  max_size               = 1
  desired_capacity       = 0
  recurrence             = "${var.recurrence}"
  autoscaling_group_name = "${aws_autoscaling_group.backup_efs_node.name}"
}
