variable "availability_zones" {
  description = "List of availability zone in which the ECS cluster should reside"
  type        = "list"
}

variable "vpc_id" {
  description = "ID of the VPC in which the ECS cluster should reside"
}

variable "subnet_ids" {
  description = "List of subnets in which the ECS cluster should reside"
  type        = "list"
}

variable "image_id" {
  description = "AMI Image ID"
  default     = "ami-7abd0209"
}

variable "instance_type" {
  description = "The instance type to use, e.g t2.small"
  default     = "t2.micro"
}

variable "key_name" {
  description = "SSH key name to use"
  default     = "Bastion"
}

variable "bastion_instances" {
  description = "Number of instances in the ASG"
  default     = "1"
}

resource "aws_security_group" "bastion" {
  name        = "Bastion"
  vpc_id      = "${var.vpc_id}"
  description = "Allows traffic from and to the Bastion EC2 instance"

  ingress {
    from_port   = 22
    to_port     = 22
    protocol    = "tcp"
    cidr_blocks = ["0.0.0.0/0"]
  }

  ingress {
    from_port   = 1194
    to_port     = 1194
    protocol    = "udp"
    cidr_blocks = ["0.0.0.0/0"]
  }


  egress {
    from_port   = 0
    to_port     = 0
    protocol    = -1
    cidr_blocks = ["0.0.0.0/0"]
  }

  tags = {
    Name        = "Bastion"
  }

  lifecycle { create_before_destroy = true }
}

data "template_file" "userdata" {
  template = "${file("${path.module}/userdata.yml")}"
  vars = {}
}

resource "aws_launch_configuration" "bastion" {
  name_prefix = "${format("%s-", "Bastion")}"

  image_id             = "${var.image_id}"
  instance_type        = "${var.instance_type}"
  key_name             = "${var.key_name}"
  security_groups      = ["${aws_security_group.bastion.id}"]
  user_data            = "${data.template_file.userdata.rendered}"

  lifecycle {
    create_before_destroy = true
    ignore_changes = [
      "image_id",
    ]
  }
}

resource "aws_autoscaling_group" "bastion" {
  name                 = "Bastion - ${aws_launch_configuration.bastion.name}"
  availability_zones   = "${var.availability_zones}"
  vpc_zone_identifier  = flatten(["${var.subnet_ids}"])
  launch_configuration = "${aws_launch_configuration.bastion.id}"
  min_size             = "${var.bastion_instances}"
  max_size             = "${var.bastion_instances}"
  desired_capacity     = "${var.bastion_instances}"

  tags {
    key = "Name"
    value = "Bastion"
    propagate_at_launch = true
  }

  tags {
    key = "service"
    value = "bastion"
    propagate_at_launch = true
  }

  lifecycle { create_before_destroy = true }
}

#############################################################################################################
# Outputs
#############################################################################################################
output "security_group" {
    value = "${aws_security_group.bastion.id}"
}
